const OpenAI = require("openai");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const articleGenerateModel = require("../DataModel/ArticleGenerateModel");
const openAiApiModel = require("../DataModel/OpenAISettingModel");
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}

const aiReWriter = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey,
  });

  const { prompt, token, randomness, presence, frequency } = req.body;
  const email = req.headers.email;
  if (prompt === "") {
    return res.status(404).json({ success: false, error: "Prompt problem" });
  }

  try {
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);

    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }

    const selectedPackage = userPackage[0];
    if (user.apiUseAiRewriterLimit >= selectedPackage.aiRewriterLimit) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for article generation" });
    }

    const response = await openai.completions.create({
      model: "gpt-3.5-turbo-instruct",
      prompt: `You are an assistant for generating articles. ${prompt}. You shall follow each of the above instructions thoroughly And then give the response.`,
      max_tokens: token,
      temperature: randomness,
      frequency_penalty: frequency,
      presence_penalty: presence,
    });

    const generatedText = response.choices[0].text;
    const timestamp = new Date().toISOString();
    const wordCount = generatedText.split(/\s+/).length;

    await articleGenerateModel.create({
      text: generatedText,
      timestamp: timestamp,
      email: email,
    });

    await userModel.findOneAndUpdate(
      { email },
      { $inc: { apiUseAiRewriterLimit: wordCount } }
    );

    res.status(200).json({ data: generatedText });
  } catch (error) {
    if (error.response) {
      console.log(error.response.status);
      console.log(error.response.data);
    } else {
      console.log(error.message);
    }
    res
      .status(400)
      .json({ success: false, error: "The text could not be generated" });
  }
};

module.exports = { aiReWriter };
