const OpenAI = require("openai");
const fs = require("fs");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const youTubeVideoAnalysesModel = require("../DataModel/youTubeModel");
const path = require("path");
const openAiApiModel = require("../DataModel/OpenAISettingModel");
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}

const aiYouTubeController = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey
  });
  const email = req.headers.email;
  try {
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }
    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);
    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }
    const selectedPackage = userPackage[0];
    if (user.apiUseYouTubeAnalyserLimit >= selectedPackage.youTubeAnalyserLimit) {
      const AUDIO_FOLDER = "./AllFile/youtubeAudios";
      fs.readdir(AUDIO_FOLDER, (err, files) => {
        if (err) {
          console.error(err);
          return res
            .status(500)
            .json({ success: false, error: "Error reading audio files" });
        }
        files.forEach((file) => {
          const filePath = path.join(AUDIO_FOLDER, file);
          fs.unlinkSync(filePath);
        });
      });

      return res
        .status(402)
        .json({ error: "Access limit exceeded for audio generation" });
    }

    const AUDIO_FOLDER = "./AllFile/youtubeAudios";
    fs.readdir(AUDIO_FOLDER, async (err, files) => {
      if (err) {
        console.error(err);
        return res
          .status(500)
          .json({ success: false, error: "Error reading audio files" });
      }
      if (files.length > 0) {
        const audioFile = `${AUDIO_FOLDER}/${files[0]}`;
        try {
          const transcription = await openai.audio.transcriptions.create({
            file: fs.createReadStream(audioFile),
            model: "whisper-1",
          });
          const text = transcription.text;
          const timestamp = new Date().toISOString();
          await userModel.findOneAndUpdate(
            { email },
            { $inc: { apiUseYouTubeAnalyserLimit : 1 } }
          );

          await youTubeVideoAnalysesModel.create({
            text: text,
            timestamp: timestamp,
            email: email,
          });

          // Delete the audio file
          files.forEach((file) => {
            const filePath = path.join(AUDIO_FOLDER, file);
            fs.unlinkSync(filePath);
          });
          res.status(200).json({ success: true, text: text });
        } catch (error) {
          res
            .status(500)
            .json({ success: false, error: "Text could not be generated" });
        }
      } else {
        res.status(500).json({ success: false, error: "No audio files found" });
      }
    });
  } catch (error) {
    console.error("Error processing request:", error);
    res
      .status(500)
      .json({ success: false, error: "Text could not be generated" });
  }
};

const youTubeAnalysesCountUserDashboard = async (req, res) => {

  const email = req.headers["email"];
  const currentYear = new Date().getFullYear();
  try {
    const result = await youTubeVideoAnalysesModel
      .aggregate([
        {
          $match: {
            email: email,
            timestamp: {
              $regex: `^${currentYear}`,
            },
          },
        },
        {
          $group: {
            _id: { $month: { $toDate: "$timestamp" } },
            Image: { $sum: 1 },
          },
        },
      ])
      .exec();
    const monthlyCounts = Array(12).fill(0);
    result.forEach((item) => {
      const monthIndex = item._id - 1;
      monthlyCounts[monthIndex] = item.Image;
    });
    res.status(200).json({
      status: "success",
      monthlyCounts: monthlyCounts,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const allYouTubeTableData = async (req, res) => {

  try {
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const matchStage = [
      {
        $match: {
          $or: [
            { url: searchRgx },
            { size: searchRgx },
            { email: searchRgx },
            { timestamp: searchRgx },
          ],
        },
      },
      {
        $addFields: {
          timestamp: {
            $toDate: "$timestamp",
          },
        },
      },
    ];

    const sortStage = {
      $sort: { timestamp: -1 },
    };

    const aggregationResult = await youTubeVideoAnalysesModel
      .aggregate([
        ...matchStage,
        sortStage,
        { $match: { email: email } },
        {
          $facet: {
            data: [{ $skip: skipRow }, { $limit: perPage }],
            total: [{ $count: "total" }],
          },
        },
      ])
      .exec();

    const data = aggregationResult[0].data || [];
    const totalCount = aggregationResult[0].total?.[0]?.total || 0;

    res.status(200).json({
      status: "success",
      data: { data, totalCount },
    });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};

const deleteYouTubeData = async (req, res) => {
  const id = req.params.id;
  const Query = { _id: id };
  try {
    const data = await youTubeVideoAnalysesModel.deleteOne(Query);
    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Not delete data", error: error });
  }
};
module.exports = {
  aiYouTubeController,
  youTubeAnalysesCountUserDashboard,
  allYouTubeTableData,
  deleteYouTubeData,
};
