const BlingDataModel = require("../DataModel/StripeDataModel");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const BlingInformationData = async (req, res) => {
  try {
    const {
      orderId,
      packageId,
      packageTime,
      packagePrice,
      packageType,
      name,
      phone,
      emailVal,
      addressVal,
      countryVal,
      cityVal,
      stateVal,
      zipVal,
      selectedVIPAssistant,
      selectedTagPremiumAssistant,
    } = req.body;

    const blingData = await BlingDataModel.create({
      orderId,
      packageId,
      packageTime,
      packagePrice,
      packageType,
      name,
      phone,
      emailVal,
      addressVal,
      countryVal,
      cityVal,
      stateVal,
      zipVal,
      selectedVIPAssistant,
      selectedTagPremiumAssistant,
    });
    res.status(200).json({ status: "success", data: blingData });
  } catch (error) {
    console.error("Error creating Bling information:", error);
    res.status(400).json({
      status: "error",
      message: "Failed to create data",
      error: error,
    });
  }
};

const TotalOrderCount = async (req, res) => {
  try {
    const resultCursor = await userModel.aggregate([
      {
        $match: {
          plan: { $ne: "Free" },
          auth: "user",
        },
      },
      {
        $sort: { Date: -1 },
      },
      {
        $facet: {
          data: [],
          total: [{ $count: "total" }],
          monthlyCounts: [
            {
              $group: {
                _id: {
                  $dateToString: {
                    format: "%Y-%m",
                    date: "$Date",
                  },
                },
                order: { $sum: 1 },
              },
            },
            { $sort: { _id: -1 } },
          ],
        },
      },
    ]);

    const result = resultCursor;
    const { data, total, monthlyCounts } = result[0];
    const formattedMonthlyCounts = Array(12).fill(0);
    monthlyCounts.forEach((item) => {
      const monthIndex = parseInt(item._id.substr(5, 2)) - 1;
      formattedMonthlyCounts[monthIndex] = item.order;
    });

    res.status(200).json({
      status: "success",
      data: data || [],
      total: total[0]?.total || 0,
      monthlyCounts: formattedMonthlyCounts || [],
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const orderList = async (req, res) => {
  try {
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const searchQuery =
      searchValue !== "0"
        ? {
            $or: [
              { orderId: searchRgx },
              { packageId: searchRgx },
              { name: searchRgx },
              { phone: searchRgx },
              { emailVal: searchRgx },
              { countryVal: searchRgx },
              { createDate: searchRgx },
              { packageTime: searchRgx },
              { packageType: searchRgx },
            ],
          }
        : {};

    const aggregatePipeline = [
      {
        $match: searchQuery,
      },
      {
        $sort: { createDate: -1 },
      },
      {
        $facet: {
          Total: [{ $count: "count" }],
          Row: [{ $skip: skipRow }, { $limit: perPage }],
        },
      },
    ];
    const [aggResult] = await BlingDataModel.aggregate(aggregatePipeline);
    const totalCount =
      aggResult.Total.length > 0 ? aggResult.Total[0].count : 0;
    const users = aggResult.Row;
    res.status(200).json({
      status: "success",
      data: { users, totalCount },
    });
  } catch (error) {
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const sentOrderId = async (req, res) => {
  try {
    const data = await BlingDataModel.find();

    if (!data) {
      return res
        .status(404)
        .json({ status: "failed", message: "User not found" });
    }
    res.status(200).json({ status: "success", data });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const updateConformData = async (req, res) => {
  const { email, packageTime, packageData, orderStatus } = req.body;

  try {
    // Update the order status based on the provided email
    const updatedOrder = await BlingDataModel.findOneAndUpdate(
      { email },
      { orderStatus },
      { new: true }
    );

    if (!updatedOrder) {
      return res.status(404).json({
        status: "failed",
        message: "Order not found with the specified email",
      });
    }

    // Find the user to update based on the email
    const user = await userModel.findOne({ email });

    if (!user) {
      return res.status(404).json({
        status: "failed",
        message: "User not found with the specified email",
      });
    }

    const packageDetailsDoc = await packageModel.findOne({
      packageDuration: packageTime,
      packageType: packageData,
    });

    if (!packageDetailsDoc) {
      return res.status(404).json({
        status: "failed",
        message: "Package not found with the specified details",
      });
    }
    const packageDetails = packageDetailsDoc.toObject();

    // Update user's plan and other package-related fields with packageDetails
    user.plan = packageDetails.packageType;
    user.packageTime = packageDetails.packageDuration;
    user.price = packageDetails.price;
    user.apiUsageImage = packageDetails.textToImage;
    user.apiUseImagination = packageDetails.imagination;
    user.apiUseFree = packageDetails.freeLimit;
    user.apiUseCaptionImage = packageDetails.imageCaption;
    user.selectedVIPAssistant = packageDetails.selectedVIP;
    user.selectedPremiumAssistant = packageDetails.selectedPremium;

    // Save the updated user data
    const savedUser = await user.save();

    res.status(200).json({
      message: "Update successful",
      data: savedUser,
      order: updatedOrder,
    });
  } catch (error) {
    console.error("Error updating user data:", error);
    res.status(500).json({
      status: "error",
      message: "An error occurred during user data update",
      error: error.message,
    });
  }
};

const deleteOrder = async (req, res) => {
  const id = req.params.id;
  const Query = { _id: id };
  try {
    const data = await BlingDataModel.deleteOne(Query);
    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Not delete data", error: error });
  }
};

const getOrderNotify = async (req, res) => {
  try {
    const total = await BlingDataModel.countDocuments({ orderStatus: "New" });

    res.status(200).json({
      status: "success",
      message: "Update successful",
      totalNewNotify: total,
    });
  } catch (error) {
    res.status(400).json({
      status: "error",
      message: "Failed to update data",
      error: error,
    });
  }
};

module.exports = {
  BlingInformationData,
  orderList,
  sentOrderId,
  updateConformData,
  deleteOrder,
  TotalOrderCount,
  getOrderNotify,
};
