const OpenAI = require("openai");
const fs = require("fs");
const audioDataModel = require("../DataModel/AudioModel");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const path = require("path");
const openAiApiModel = require("../DataModel/OpenAISettingModel");
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}

const AUDIO_FOLDER = "./AllFile/AudioFiles";
const getLastFileNumber = (files) => {
  return files
    .filter((fileName) => /^\d+\..+$/.test(fileName))
    .map((fileName) => {
      const parts = fileName.split(".");
      return {
        number: parseInt(parts[0], 10),
        extension: parts[1],
      };
    })
    .sort((a, b) => a.number - b.number)
    .pop();
};

const SpeechToText = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey,
  });

  const email = req.headers.email;
  try {
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);

    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }

    const selectedPackage = userPackage[0];
    if (user.apiUseSpeechToTextLimit >= selectedPackage.speechToTextLimit) {
      fs.readdir(AUDIO_FOLDER, (err, files) => {
        if (err) {
          console.error(err);
          return res
            .status(500)
            .json({ success: false, error: "Error reading audio files" });
        }
        files.forEach((file) => {
          const filePath = path.join(AUDIO_FOLDER, file);
          fs.unlinkSync(filePath);
        });
      });

      return res
        .status(402)
        .json({ error: "Access limit exceeded for audio generation" });
    }

    fs.readdir(AUDIO_FOLDER, async (err, files) => {
      if (err) {
        console.error(err);
        return res
          .status(500)
          .json({ success: false, error: "Error reading audio files" });
      }

      const lastFile = getLastFileNumber(files);
      if (lastFile !== undefined) {
        const lastFileName = `${lastFile.number}.${lastFile.extension}`;
        const lastFilePath = path
          .join(AUDIO_FOLDER, lastFileName)
          .replace(/\\/g, "/");
        try {
          const transcription = await openai.audio.transcriptions.create({
            file: fs.createReadStream(lastFilePath),
            model: "whisper-1",
          });

          const Text = transcription.text;
          const timestamp = new Date().toISOString();

          await audioDataModel.create({
            Text: Text,
            email: email,
            createDate: timestamp,
          });

          await userModel.findOneAndUpdate(
            { email },
            { $inc: { apiUseSpeechToTextLimit: 1 } }
          );
          // Delete the content of the folder
          files.forEach((file) => {
            const filePath = path.join(AUDIO_FOLDER, file);
            fs.unlinkSync(filePath);
          });

          res.status(200).json({ success: true, text: Text });
        } catch (error) {
          res
            .status(500)
            .json({ success: false, error: "Text could not be generated" });
        }
      } else {
        res.status(500).json({ success: false, error: "No audio files found" });
      }
    });
  } catch (error) {
    res
      .status(500)
      .json({ success: false, error: "Text could not be generated" });
  }
};

const formatTimestamp = (createDate) => {
  const date = new Date(createDate);
  const day = date.getDate().toString().padStart(2, "0");
  const monthNames = [
    "Jan",
    "Feb",
    "Mar",
    "Apr",
    "May",
    "Jun",
    "Jul",
    "Aug",
    "Sep",
    "Oct",
    "Nov",
    "Dec",
  ];
  const month = monthNames[date.getMonth()];
  return `${day} ${month}`;
};

const AllFileAudioText = async (req, res) => {
  const email = req.headers["email"];
  const currentDate = new Date();
  const currentYear = currentDate.getFullYear();
  const currentMonth = currentDate.getMonth() + 1;
  try {
    const resultCursor = audioDataModel.aggregate([
      {
        $match: {
          email: email,
          createDate: {
            $regex: `^${currentYear}-${
              currentMonth < 10 ? "0" + currentMonth : currentMonth
            }-\\d{2}T\\d{2}:\\d{2}:\\d{2}.\\d{3}Z$`,
          },
        },
      },
      {
        $addFields: {
          timestamp: "$createDate",
        },
      },
      {
        $sort: { timestamp: -1 },
      },
      {
        $facet: {
          data: [],
          total: [{ $count: "total" }],
          dailyCounts: [
            {
              $group: {
                _id: {
                  $dateToString: {
                    format: "%m %d",
                    date: { $toDate: "$createDate" },
                  },
                },
                Text: { $sum: 1 },
              },
            },
            { $sort: { _id: -1 } },
          ],
        },
      },
    ]);
    const result = await resultCursor.exec();
    const { data, total, dailyCounts } = result[0];
    const formattedDailyCounts = dailyCounts.map((item) => ({
      ...item,
      _id: formatTimestamp(item._id),
    }));
    res.status(200).json({
      status: "success",
      data: data,
      total: total[0]?.total || 0,
      dailyCounts: formattedDailyCounts || [],
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};

module.exports = { SpeechToText, AllFileAudioText };
