const codeModel = require("../DataModel/CodeModel");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const OpenAI = require("openai");
const openAiApiModel = require("../DataModel/OpenAISettingModel");
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}

const AICodeGenerate = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey,
  });
  const prompt = req.body.prompt;
  const email = req.headers.email;
  if (prompt === "") {
    return res.status(404).json({ success: false, error: "Prompt problem" });
  }

  try {
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);

    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }

    const selectedPackage = userPackage[0];
    if (user.apiUseAiCodeGenerateLimit >= selectedPackage.aiCodeGenerateLimit) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for article generation" });
    }

    const response = await openai.chat.completions.create({
      model: "gpt-3.5-turbo",
      messages: [
        {
          role: "system",
          content: `You are a code generator. You must answer only in markdown code snippets.code will be decorate,untestable,clean.
         you will write only code snippets `,
        },
        { role: "user", content: `${prompt}` },
      ],
      max_tokens: 3000,
      temperature: 0.7,
      frequency_penalty: 0.4,
      presence_penalty: 0.3,
    });
    let generatedText = String(response.choices[0].message.content);
    const timestamp = new Date().toISOString();
    const wordCount = generatedText.split(/\s+/).length;

    await codeModel.create({
      code: generatedText,
      timestamp: timestamp,
      email: email,
    });

    await userModel.findOneAndUpdate(
      { email },
      { $inc: { apiUseAiCodeGenerateLimit: wordCount } }
    );

    res.status(200).json({ data: generatedText });
  } catch (error) {
    if (error.response) {
      console.log(error.response.status);
      console.log(error.response.data);
    } else {
      console.log(error.message);
    }
    res
      .status(400)
      .json({ success: false, error: "The text could not be generated" });
  }
};

const AiCodeList = async (req, res) => {
  try {
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const matchStage = [
      {
        $match: {
          $or: [
            { url: searchRgx },
            { size: searchRgx },
            { email: searchRgx },
            { timestamp: searchRgx },
          ],
        },
      },
      {
        $addFields: {
          timestamp: {
            $toDate: "$timestamp",
          },
        },
      },
    ];
    const sortStage = {
      $sort: { timestamp: -1 },
    };

    const aggregationResult = await codeModel
      .aggregate([
        ...matchStage,
        sortStage,
        { $match: { email: email } },
        {
          $facet: {
            data: [{ $skip: skipRow }, { $limit: perPage }],
            total: [{ $count: "total" }],
          },
        },
      ])
      .exec();

    const data = aggregationResult[0].data || [];
    const totalCount = aggregationResult[0].total?.[0]?.total || 0;

    res.status(200).json({
      status: "success",
      data: { data, totalCount },
    });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};
const deleteCodeDocument = async (req, res) => {
  const id = req.params.id;
  const Query = { _id: id };
  try {
    const data = await codeModel.deleteOne(Query);
    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Not delete data", error: error });
  }
};
module.exports = { AICodeGenerate, AiCodeList, deleteCodeDocument };
