const OpenAI = require("openai");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const articleGenerateModel = require("../DataModel/ArticleGenerateModel");
const openAiApiModel = require("../DataModel/OpenAISettingModel");

// Function to asynchronously retrieve OpenAI API key
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}
const ArticleGenerate = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey,
  });
  const { prompt, token, randomness } = req.body;

  const email = req.headers.email;

  if (prompt === "") {
    return res.status(404).json({ success: false, error: "Prompt problem" });
  }

  try {
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);

    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }

    const selectedPackage = userPackage[0];
    if (user.apiUseAiTemplateLimit >= selectedPackage.aiTemplateLimit) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for article generation" });
    }

    const response = await openai.completions.create({
      model: "gpt-3.5-turbo-instruct",
      prompt: `You are an assistant for generating articles. ${prompt}. You shall follow each of the above instructions thoroughly And then give the response.`,
      max_tokens: token,
      temperature: randomness,
    });

    const generatedText = response.choices[0].text;
    const timestamp = new Date().toISOString();
    const wordCount = generatedText.split(/\s+/).length;

    await articleGenerateModel.create({
      text: generatedText,
      timestamp: timestamp,
      email: email,
    });

    await userModel.findOneAndUpdate(
      { email },
      { $inc: { apiUseAiTemplateLimit: wordCount } }
    );

    res.status(200).json({ data: generatedText });
  } catch (error) {
    if (error.response) {
      console.log(error.response.status);
      console.log(error.response.data);
    } else {
      console.log(error.message);
    }
    res
      .status(400)
      .json({ success: false, error: "The text could not be generated" });
  }
};

const articleCountUserDashboard = async (req, res) => {
  const email = req.headers["email"];
  const currentYear = new Date().getFullYear();

  try {
    const result = await articleGenerateModel
      .aggregate([
        {
          $match: {
            email: email,
            timestamp: {
              $regex: `^${currentYear}`,
            },
          },
        },
        {
          $group: {
            _id: { $month: { $toDate: "$timestamp" } },
            totalWords: { $sum: { $size: { $split: ["$text", " "] } } },
          },
        },
        {
          $group: {
            _id: null,
            yearlyTotalWords: { $sum: "$totalWords" },
            totalWordsAllTime: { $sum: "$totalWords" },
            monthlyCounts: { $push: "$$ROOT" },
          },
        },
      ])
      .exec();

    if (result.length === 0) {
      return res.status(200).json({
        status: "success",
        totalWordsAllTime: 0,
        monthlyCounts: Array(12).fill(0),
      });
    }

    const totalWordsAllTime = result[0].totalWordsAllTime;
    const monthlyCounts = Array(12).fill(0);
    result[0].monthlyCounts.forEach((item) => {
      const monthIndex = item._id - 1;
      monthlyCounts[monthIndex] = item.totalWords;
    });

    res.status(200).json({
      status: "success",
      totalWordsAllTime: totalWordsAllTime,
      monthlyCounts: monthlyCounts,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const allArticleTableData = async (req, res) => {
  try {
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const matchStage = [
      {
        $match: {
          $or: [
            { url: searchRgx },
            { size: searchRgx },
            { email: searchRgx },
            { timestamp: searchRgx },
          ],
        },
      },
      {
        $addFields: {
          timestamp: {
            $toDate: "$timestamp",
          },
        },
      },
    ];

    const sortStage = {
      $sort: { timestamp: -1 },
    };

    const aggregationResult = await articleGenerateModel
      .aggregate([
        ...matchStage,
        sortStage,
        { $match: { email: email } },
        {
          $facet: {
            data: [{ $skip: skipRow }, { $limit: perPage }],
            total: [{ $count: "total" }],
          },
        },
      ])
      .exec();

    const data = aggregationResult[0].data || [];
    const totalCount = aggregationResult[0].total?.[0]?.total || 0;

    res.status(200).json({
      status: "success",
      data: { data, totalCount },
    });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};
const deleteArticleDocument = async (req, res) => {
  const id = req.params.id;
  const Query = { _id: id };
  try {
    const data = await articleGenerateModel.deleteOne(Query);
    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Not delete data", error: error });
  }
};

module.exports = {
  ArticleGenerate,
  articleCountUserDashboard,
  allArticleTableData,
  deleteArticleDocument,
};
