const assistantModel = require("../DataModel/AssistantModel");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const AiAssistAdvanceSettingModel = require("../DataModel/AssistantAdvanceModel");
const openAiApiModel = require("../DataModel/OpenAISettingModel");
const axios = require("axios");

async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}

const chatWithOpenAI = async (req, res) => {
  try {
    const apiKey = await getOpenAIKey();
    const { id, chatData } = req.body;
    const email = req.headers.email;

    let assistantAdvanceModel = await AiAssistAdvanceSettingModel.findOne({
      selectedId: id,
      userEmail: email,
    });
    if (!assistantAdvanceModel) {
      const defaultAssistantModelData = await assistantModel.findOne({
        _id: id,
      });
      if (!defaultAssistantModelData) {
        return res
          .status(400)
          .json({ error: "Default assistant data not found" });
      }
      assistantAdvanceModel = {
        selectedModel: defaultAssistantModelData.selectedModel,
        promptText: defaultAssistantModelData.promptDescription,
        token: defaultAssistantModelData.token,
        randomness: defaultAssistantModelData.randomness,
        frequencyPenalty: defaultAssistantModelData.frequencyPenalty,
        presencePenalty: defaultAssistantModelData.presencePenalty,
      };
    }

    // Construct API request body
    const apiRequestBody = {
      model: assistantAdvanceModel.selectedModel,
      messages: [
        {
          role: "system",
          content: `Must fallow,Response Format: Your sagacity flows through an HTML document, Title will <h2>Title answer</h2> tag, 
        subtitle will <h4>Sub Title answer</h4> tag 
        and text will <p>Text</p>${assistantAdvanceModel.promptText}`,
        },
        { role: "user", content: `${chatData}` },
      ],
      max_tokens: assistantAdvanceModel.token,
      temperature: assistantAdvanceModel.randomness,
      frequency_penalty: assistantAdvanceModel.frequencyPenalty,
      presence_penalty: assistantAdvanceModel.presencePenalty,
    };

    // Validate user existence
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    // Fetch user package
    const userPackage = await packageModel.findOne({
      packageType: user.plan,
      packageDuration: user.packageTime,
    });

    if (!userPackage) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }

    // Check if user has exceeded chatbot interaction limit
    if (user.apiUseAiChatAssistantLimit >= userPackage.aiChatAssistantLimit) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for chatbot interaction" });
    }

    const response = await axios.post(
      "https://api.openai.com/v1/chat/completions",
      apiRequestBody,
      {
        headers: {
          Authorization: `Bearer ${apiKey}`,
          "Content-Type": "application/json",
        },
      }
    );
    const firstResponse = response.data.choices[0].message.content;
    const timestamp = new Date().toISOString();
    const assistant = await assistantModel.findOne({ _id: id });

    if (!assistant) {
      return res.status(400).json({ error: "Assistant not found" });
    }
    const newChatMessage = {
      message: firstResponse,
      sender: "chatGpt",
      sendTime: timestamp,
      email: email,
      inputValue: chatData,
      matchId: id,
    };

    assistant.last_chat.push(newChatMessage);
    await assistant.save();
    const wordCount = firstResponse.split(/\s+/).length;
    await userModel.findOneAndUpdate(
      { email },
      { $inc: { apiUseAiChatAssistantLimit: wordCount } }
    );

    res.status(200).json({ data: firstResponse });
  } catch (error) {
    console.error(error.message);
    res
      .status(500)
      .json({ error: "Error in the API request", details: error.message });
  }
};

const allAssistantTableUserData = async (req, res) => {

  try {
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const matchStage = [
      {
        $match: {
          $or: [{ "last_chat.message": searchRgx }],
        },
      },
      {
        $addFields: {
          createDate: {
            $toDate: "$createDate",
          },
        },
      },
    ];

    const sortStage = {
      $sort: { createDate: -1 },
    };

    const aggregationResult = await assistantModel
      .aggregate([
        ...matchStage,
        sortStage,
        {
          $match: { "last_chat.email": email },
        },
        {
          $facet: {
            data: [
              { $unwind: "$last_chat" },
              { $match: { "last_chat.email": email } },
              {
                $group: {
                  _id: null,
                  data: { $push: "$last_chat" },
                  count: { $sum: 1 },
                },
              },
              { $addFields: { skipRow: skipRow } },
              {
                $project: {
                  data: { $slice: ["$data", skipRow, perPage] },
                  count: 1,
                },
              },
            ],
            total: [{ $count: "total" }],
          },
        },
      ])
      .exec();

    const lastChatMatched = aggregationResult[0].data?.[0]?.data || [];
    const lastChatCount = aggregationResult[0].data?.[0]?.count || 0;
    const totalCount = aggregationResult[0].total?.[0]?.total || 0;

    res.status(200).json({
      status: "success",
      data: { lastChatMatched, lastChatCount, totalCount },
    });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};

const createAssistant = async (req, res) => {
  const {
    assistantIcon,
    assistantName,
    title,
    category,
    packageType,
    promptDescription,
    brandIcon,
    selectedModel,
    token,
    randomness,
    frequencyPenalty,
    presencePenalty,
  } = req.body;
  try {
    const package = await assistantModel.create({
      assistantIcon,
      assistantName,
      title,
      category,
      packageType,
      promptDescription,
      brandIcon,
      selectedModel,
      token,
      randomness,
      frequencyPenalty,
      presencePenalty,
    });

    res.status(200).json({ status: "success", data: package });
  } catch (error) {
    res
      .status(500)
      .json({ status: "failed", message: "An error occurred", error: error });
    console.log(error);
  }
};
const createAssistantAdvanceSetting = async (req, res) => {
  const {
    userEmail,
    selectedId,
    randomness,
    frequencyPenalty,
    presencePenalty,
    promptText,
    token,
    selectedModel,
  } = req.body;
  try {
    let existingSetting = await AiAssistAdvanceSettingModel.findOne({
      selectedId,
    });
    if (!existingSetting) {
      const newSetting = new AiAssistAdvanceSettingModel({
        userEmail,
        selectedId,
        randomness,
        frequencyPenalty,
        presencePenalty,
        promptText,
        token,
        selectedModel,
      });
      const savedSetting = await newSetting.save();
      res.status(200).json({ status: "success", data: savedSetting });
    } else {
      res.status(200).json({ status: "success", data: existingSetting });
    }
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: "failed",
      message: "An error occurred",
      error: error.message,
    });
  }
};

const updateAssistantData = async (req, res) => {
  try {
    const {
      id,
      assistantIcon,
      assistantName,
      title,
      category,
      packageType,
      promptDescription,
      brandIcon,
      selectedModel,
      token,
      randomness,
      frequencyPenalty,
      presencePenalty,
    } = req.body;

    const updateAssistantData = {
      id: id,
      assistantIcon: assistantIcon,
      assistantName: assistantName,
      title: title,
      category: category,
      packageType: packageType,
      promptDescription: promptDescription,
      brandIcon: brandIcon,
      selectedModel,
      token,
      randomness,
      frequencyPenalty,
      presencePenalty,
    };
    const data = await assistantModel.findByIdAndUpdate(
      id,
      updateAssistantData,
      {
        new: true,
      }
    );
    res.status(200).json({ message: "success", data: data });
  } catch (error) {
    console.error("Error updating blog post:", error);
    res.status(400).json({
      status: "error",
      message: "Failed to update data",
      error: error,
    });
  }
};
const updateAssistantSettingModel = async (req, res) => {
  try {
    const {
      userEmail,
      selectedId,
      promptText,
      selectedModel,
      token,
      randomness,
      frequencyPenalty,
      presencePenalty,
    } = req.body;
    const updateAssistantData = {
      userEmail: userEmail,
      promptText: promptText,
      selectedModel: selectedModel,
      token: token,
      randomness: randomness,
      frequencyPenalty: frequencyPenalty,
      presencePenalty: presencePenalty,
    };
    const data = await AiAssistAdvanceSettingModel.findOneAndUpdate(
      { selectedId: selectedId },
      updateAssistantData,
      {
        new: true,
      }
    );

    if (!data) {
      return res.status(404).json({
        status: "error",
        message: "Assistant settings not found",
      });
    }

    res.status(200).json({ message: "success", data: data });
  } catch (error) {
    console.error("Error updating assistant settings:", error);
    res.status(400).json({
      status: "error",
      message: "Failed to update data",
      error: error,
    });
  }
};
const assistantListAdmin = async (req, res) => {
  try {
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const searchQuery =
      searchValue !== "0"
        ? {
            $or: [
              { orderId: searchRgx },
              { packageId: searchRgx },
              { name: searchRgx },
              { phone: searchRgx },
              { emailVal: searchRgx },
              { countryVal: searchRgx },
              { createDate: searchRgx },
              { packageTime: searchRgx },
              { packageType: searchRgx },
            ],
          }
        : {};

    const [totalCount, users] = await Promise.all([
      assistantModel.countDocuments(searchQuery),
      assistantModel.find(searchQuery).skip(skipRow).limit(perPage),
    ]);

    res.status(200).json({
      status: "success",
      data: { users, totalCount },
    });
  } catch (error) {
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const showHideAssistant = async (req, res) => {
  const { id, bullionData } = req.body;
  try {
    const matchId = { _id: id };

    const updateData = await assistantModel.updateOne(matchId, {
      $set: { active: bullionData },
    });
    if (updateData.nModified === 0) {
      return res.status(404).json({
        status: "failed",
        message: "User not found or no changes applied",
      });
    }
    res.status(200).json({ message: "Update successful" });
  } catch (error) {
    console.error("Error updating data:", error);
    res.status(500).json({
      status: "error",
      message: "An error occurred during data update",
    });
  }
};

const getEditAssistantAdvanceModel = async (req, res) => {
  const { id } = req.body;

  try {
    const data = await AiAssistAdvanceSettingModel.findOne({ selectedId: id });

    if (!data) {
      return res
        .status(404)
        .json({ status: "error", message: "Data not found" });
    }
    res.status(200).json({
      status: "success",
      message: "Data found successfully",
      data: data,
    });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Failed to fetch data", error: error });
  }
};
const deleteAssistant = async (req, res) => {
  const id = req.params.id;
  const Query = { _id: id };
  try {
    const data = await assistantModel.deleteOne(Query);
    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Not delete data", error: error });
  }
};
const getAllAssistantUser = async (req, res) => {
  try {
    const data = await assistantModel.find({ active: "true" });
    const SpecialistAssistant = data.filter(
      (item) => item.category === "Specialist"
    );
    const EducationAssistant = data.filter(
      (item) => item.category === "Education"
    );
    const HealthAssistant = data.filter((item) => item.category === "Health");
    const BusinessAssistant = data.filter(
      (item) => item.category === "Business"
    );
    const InstructorAssistant = data.filter(
      (item) => item.category === "Instructor"
    );
    const otherAssistant = data.filter((item) => item.category === "other");

    res.status(200).json({
      success: true,
      SpecialistAssistant,
      EducationAssistant,
      HealthAssistant,
      BusinessAssistant,
      InstructorAssistant,
      otherAssistant,
      data,
    });
  } catch (error) {
    res.status(500).json({ success: false, message: "Internal Server Error" });
  }
};

const getSingleAssistantData = async (req, res) => {
  const id = req.params.id;
  try {
    const data = await assistantModel.findById(id);
    if (!data) {
      return res
        .status(404)
        .json({ status: "error", message: "Data not found" });
    }
    res.status(200).json({
      status: "success",
      message: "Data found successfully",
      data: data,
    });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Failed to fetch data", error: error });
  }
};
const setFavoriteAssistantData = async (req, res) => {
  const favoriteData = req.body.favoriteData;
  const id = req.body.id;
  try {
    const matchId = { _id: id };
    const updateData = await assistantModel.updateOne(matchId, {
      $set: { favorite: favoriteData },
    });
    if (updateData.nModified === 0) {
      return res.status(404).json({
        status: "failed",
        message: "User not found or no changes applied",
      });
    }
    res.status(200).json({ message: "Update successful" });
  } catch (error) {
    console.error("Error updating data:", error);
    res.status(500).json({
      status: "error",
      message: "An error occurred during data update",
    });
  }
};
const getAllFavoriteAssistant = async (req, res) => {
  try {
    const data = await assistantModel.find({ favorite: "like" });
    res.status(200).json({ success: true, data: data });
  } catch (error) {
    res.status(500).json({ success: false, message: "Internal Server Error" });
  }
};

const updateAssistantPositionInDB = async (req, res) => {
  const { id, newPosition } = req.body;
  try {
    const assistant = await assistantModel.findById(id);
    assistant.position = newPosition;
    await assistant.save();
    res
      .status(200)
      .json({ message: "Assistant position updated successfully" });
  } catch (error) {
    console.error("Error updating assistant position:", error);
    res.status(500).json({ error: "Failed to update assistant position" });
  }
};

const deleteChatDocument = async (req, res) => {
  const id = req.params.id;

  try {
    const result = await assistantModel.updateOne(
      { "last_chat._id": id },
      { $pull: { last_chat: { _id: id } } }
    );
    if (result.nModified === 0) {
      return res.status(404).json({
        status: "error",
        message: "No document found with the specified _id",
      });
    }

    res
      .status(200)
      .json({ status: "success", message: "Message deleted successfully" });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};
const vipAssistantListAdmin = async (req, res) => {
  try {
    const data = await assistantModel.find({ packageType: "VIP" });
    res.status(200).json({ success: true, data: data });
  } catch (error) {
    res.status(500).json({ success: false, message: "Internal Server Error" });
  }
};

const premiumAssistantListAdmin = async (req, res) => {
  try {
    const data = await assistantModel.find({ packageType: "Premium" });
    res.status(200).json({ success: true, data: data });
  } catch (error) {
    res.status(500).json({ success: false, message: "Internal Server Error" });
  }
};

module.exports = {
  createAssistantAdvanceSetting,
  createAssistant,
  assistantListAdmin,
  showHideAssistant,
  getEditAssistantAdvanceModel,
  updateAssistantData,
  updateAssistantSettingModel,
  deleteAssistant,
  getAllAssistantUser,
  getSingleAssistantData,
  setFavoriteAssistantData,
  getAllFavoriteAssistant,
  chatWithOpenAI,
  allAssistantTableUserData,
  deleteChatDocument,
  updateAssistantPositionInDB,
  vipAssistantListAdmin,
  premiumAssistantListAdmin,
};
