const natural = require("natural");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");

function analyzeLanguageComplexity(text) {
  const tokenizer = new natural.WordTokenizer();
  const sentences = tokenizer.tokenize(text);
  const averageSentenceLength =
    sentences.reduce((total, sentence) => total + sentence.length, 0) /
    sentences.length;
  const vocabularyRichness = new Set(tokenizer.tokenize(text)).size;
  const syntacticComplexity = sentences.length;
  return { averageSentenceLength, vocabularyRichness, syntacticComplexity };
}

function analyzeContextUnderstanding(text) {
  return Math.random() >= 0.5;
}

function analyzeKnowledgeBasedFeatures(text) {
  return Math.random() >= 0.5;
}

function detectAuthorship(sentence) {
  const { averageSentenceLength, vocabularyRichness, syntacticComplexity } =
    analyzeLanguageComplexity(sentence);
  const hasGoodContextUnderstanding = analyzeContextUnderstanding(sentence);
  const hasKnowledgeBasedFeatures = analyzeKnowledgeBasedFeatures(sentence);
  if (
    averageSentenceLength > 20 &&
    vocabularyRichness > 100 &&
    syntacticComplexity > 5
  ) {
    return "Human";
  } else if (hasGoodContextUnderstanding && hasKnowledgeBasedFeatures) {
    return "Human";
  } else {
    return "AI";
  }
}

const detectContent = async (req, res) => {
  const { text } = req.body;
  const { email } = req.headers;

  try {
    // Validate input text
    if (!text) {
      return res
        .status(400)
        .json({ error: "Text input is missing in the request body" });
    }

    // Find user
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    // Find user's package
    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);
    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }
    const selectedPackage = userPackage[0];

    if (user.apiUseContentDetector >= selectedPackage.aiContentDetector) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for content detection" });
    }

    const tokenizer = new natural.SentenceTokenizer();
    const sentences = tokenizer.tokenize(text);
    const result = sentences.map((sentence) => detectAuthorship(sentence));
    const humanCount = result.filter(
      (authorship) => authorship === "Human"
    ).length;
    const aiCount = result.length - humanCount;
    const totalSentences = result.length;
    const humanPercentage = (humanCount / totalSentences) * 100;
    const aiPercentage = (aiCount / totalSentences) * 100;
    const overallResult =
      humanCount > aiCount
        ? { result: "Human", percentage: humanPercentage }
        : { result: "AI", percentage: aiPercentage };

    await userModel.findOneAndUpdate(
      { email },
      { $inc: { apiUseContentDetector: 1 } }
    );

    res.json({ overallResult });
  } catch (error) {
    console.error("Error in content detection:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};

module.exports = { detectContent };
