const OpenAI = require("openai");
const fs = require("fs");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const editAudioModel = require("../DataModel/EditAudioModel");
const path = require("path");
const openAiApiModel = require("../DataModel/OpenAISettingModel");
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}
const AUDIO_FOLDER = "./AllFile/EditAudioFile";

const editAudioFile = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey,
  });
  const { inputText, selectedLanguage, selectedVoice, voiceoverModel } =
    req.body;
  const email = req.headers.email;
  try {
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }
    const userPackage = await packageModel.findOne({
      packageType: user.plan,
      packageDuration: user.packageTime,
    });

    if (!userPackage) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }

    if (user.apiUseEditAudioLimit >= userPackage.editAudioLimit) {
      fs.readdir(AUDIO_FOLDER, (err, files) => {
        if (err) {
          console.error(err);
          return res
            .status(500)
            .json({ success: false, error: "Error reading audio files" });
        }
        files.forEach((file) => {
          const filePath = path.join(AUDIO_FOLDER, file);
          fs.unlinkSync(filePath);
        });
      });

      return res
        .status(402)
        .json({ error: "Access limit exceeded for audio generation" });
    }

    fs.readdir(AUDIO_FOLDER, async (err, files) => {
      if (err) {
        console.error(err);
        return res
          .status(500)
          .json({ success: false, error: "Error reading audio files" });
      }

      const lastFile = getLastFileNumber(files);
      if (lastFile !== undefined) {
        const lastFileName = `${lastFile.number}.${lastFile.extension}`;
        const lastFilePath = path
          .join(AUDIO_FOLDER, lastFileName)
          .replace(/\\/g, "/");

        try {
          const transcription = await openai.audio.transcriptions.create({
            file: fs.createReadStream(lastFilePath),
            model: "whisper-1",
          });

          const Text = transcription.text;

          //Delete the content of the folder
          files.forEach((file) => {
            const filePath = path.join(AUDIO_FOLDER, file);
            fs.unlinkSync(filePath);
          });

          const mp3 = await openai.audio.speech.create({
            model: voiceoverModel,
            voice: selectedVoice,
            input: `${inputText} ${Text}. must generate audio by this language ${selectedLanguage}`,
          });

          const fileName = `${Date.now()}.mp3`;
          const filePath = path.join("./AllFile/EditAudioFile", fileName);
          const buffer = Buffer.from(await mp3.arrayBuffer());

          await fs.promises.writeFile(filePath, buffer);
          await fs.promises.access(filePath);

          const relativePath = `/findEditAudio/${fileName}`;
          const timestamp = new Date().toISOString();

          await userModel.findOneAndUpdate(
            { email },
            { $inc: { apiUseEditAudioLimit: 1 } }
          );

          const result = await editAudioModel.create({
            audio: relativePath,
            text: inputText,
            createDate: timestamp,
            email,
            language: selectedVoice,
            model: voiceoverModel,
            voice: selectedVoice,
          });
          res.status(200).json({ success: true, text: result });
        } catch (error) {
          res
            .status(500)
            .json({ success: false, error: "Text could not be generated" });
        }
      } else {
        res.status(500).json({ success: false, error: "No audio files found" });
      }
    });
  } catch (error) {
    res
      .status(500)
      .json({ success: false, error: "Text could not be generated" });
  }
};

const getLastFileNumber = (files) => {
  return files
    .filter((fileName) => /^\d+\..+$/.test(fileName))
    .map((fileName) => {
      const parts = fileName.split(".");
      return {
        number: parseInt(parts[0], 10),
        extension: parts[1],
      };
    })
    .sort((a, b) => a.number - b.number)
    .pop();
};

const allEditAudio = async (req, res) => {
  try {
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const matchStage = [
      {
        $match: {
          $or: [
            { url: searchRgx },
            { size: searchRgx },
            { email: searchRgx },
            { timestamp: searchRgx },
          ],
        },
      },
      {
        $addFields: {
          timestamp: {
            $toDate: "$timestamp",
          },
        },
      },
    ];

    const sortStage = {
      $sort: { timestamp: -1 },
    };

    const aggregationResult = await editAudioModel
      .aggregate([
        ...matchStage,
        sortStage,
        { $match: { email: email } },
        {
          $facet: {
            data: [{ $skip: skipRow }, { $limit: perPage }],
            total: [{ $count: "total" }],
          },
        },
      ])
      .exec();

    const data = aggregationResult[0].data || [];
    const totalCount = aggregationResult[0].total?.[0]?.total || 0;

    res.status(200).json({
      status: "success",
      data: { data, totalCount },
    });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};

module.exports = { editAudioFile, allEditAudio };
