const fs = require("fs");
const path = require("path");
const base64 = require("base64-js");
const axios = require("axios");
const OpenAI = require("openai");
const CaptionImageModel = require("../DataModel/CaptionImageModel");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const openAiApiModel = require("../DataModel/OpenAISettingModel");

// Function to asynchronously retrieve OpenAI API key
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}

const imageCaptionGenerate = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey,
  });

  const { text } = req.body;
  const { email } = req.headers;
  try {
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);

    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }

    const selectedPackage = userPackage[0];
    if (user.apiUseImageCaption >= selectedPackage.imageCaption) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for image generation" });
    }
    const CaptionImagesFolder = "./AllFile/CaptionImages";
    fs.readdir(CaptionImagesFolder, async (err, files) => {
      if (err) {
        console.error(err);
        return res
          .status(500)
          .json({ success: false, error: "Error reading image" });
      }

      if (files.length === 0) {
        return res
          .status(404)
          .json({ success: false, error: "No files found in the folder" });
      }

      const lastFile = files[files.length - 1];
      const imagePath = path.join(CaptionImagesFolder, lastFile);
      const captionImage = "captionImage";
      const captionImageUrl = `${captionImage}/${lastFile}`;
      const encodeImage = (imagePath) => {
        return new Promise((resolve, reject) => {
          fs.readFile(imagePath, (err, data) => {
            if (err) reject(err);
            else resolve(base64.fromByteArray(data));
          });
        });
      };
      const base64Image = await encodeImage(imagePath);
      const headers = {
        "Content-Type": "application/json",
        Authorization: `Bearer ${openai.apiKey}`,
      };
      const payload = {
        model: "gpt-4o",
        messages: [
          {
            role: "user",
            content: [
              {
                type: "text",
                text: `your target is make image caption and description of image with professionalism. ${text}. 
                must you will generate image caption and description and must caption hold on <h2>caption title</h2> and description hold on <p>description text</p>`,
              },
              {
                type: "image_url",
                image_url: {
                  url: `data:image/jpeg;base64,${base64Image}`,
                },
              },
            ],
          },
        ],
        max_tokens: 1200,
      };

      const response = await axios.post(
        "https://api.openai.com/v1/chat/completions",
        payload,
        { headers }
      );
      const responseData = await response.data;
      if (responseData.choices && responseData.choices.length > 0) {
        const messages = responseData.choices.map(
          (choice) => choice.message.content
        );
        const finalResult = messages[0];
        const timestamp = new Date().toISOString();
        await CaptionImageModel.create({
          text: finalResult,
          timestamp: timestamp,
          email: email,
          image: captionImageUrl,
        });

        await userModel.findOneAndUpdate(
          { email },
          {
            $inc: { apiUseImageCaption: 1 },
          }
        );
        res.status(200).json({ success: true, messages: finalResult });
      } else {
        res.status(200).json({ success: false, error: "No messages found" });
      }
    });
  } catch (error) {
    console.error(error);
    res
      .status(400)
      .json({ success: false, error: "The text could not be generated" });
  }
};

const captionImageDataList = async (req, res) => {
  try {
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const matchStage = [
      {
        $match: {
          $or: [
            { url: searchRgx },
            { size: searchRgx },
            { email: searchRgx },
            { timestamp: searchRgx },
          ],
        },
      },
      {
        $addFields: {
          timestamp: {
            $toDate: "$timestamp",
          },
        },
      },
    ];

    const sortStage = {
      $sort: { timestamp: -1 },
    };

    const aggregationResult = await CaptionImageModel.aggregate([
      ...matchStage,
      sortStage,
      { $match: { email: email } },
      {
        $facet: {
          data: [{ $skip: skipRow }, { $limit: perPage }],
          total: [{ $count: "total" }],
        },
      },
    ]).exec();

    const data = aggregationResult[0].data || [];
    const totalCount = aggregationResult[0].total?.[0]?.total || 0;

    res.status(200).json({
      status: "success",
      data: { data, totalCount },
    });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};

const captionImageData = async (req, res) => {
  try {
    const { email } = req.headers;
    const data = await CaptionImageModel.find({ email });
    res.json(data);
  } catch (err) {
    console.error("Error fetching captioned images:", err);
    res.status(500).json({ error: "An error occurred while fetching data" });
  }
};
const deleteCaptionData = async (req, res) => {
  const id = req.params.id;
  const Query = { _id: id };
  try {
    const data = await CaptionImageModel.deleteOne(Query);
    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Not delete data", error: error });
  }
};

module.exports = {
  imageCaptionGenerate,
  captionImageDataList,
  captionImageData,
  deleteCaptionData,
};
