const fs = require("fs").promises;
const path = require("path");
const base64 = require("base64-js");
const OpenAI = require("openai");
const axios = require('axios');
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const imageToAudioModel = require("../DataModel/ImageToAudio");
const openAiApiModel = require("../DataModel/OpenAISettingModel");

// Function to asynchronously retrieve OpenAI API key
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}
const imageToAudioCreate = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey,
  });

  try {
    const { descriptionValue, selectedLanguage } = req.body;
    const email = req.headers.email;
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }
    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);
    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }

    const selectedPackage = userPackage[0];
    if (user.apiUseImageAudioLimit >= selectedPackage.imageAudioLimit) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for image generation" });
    }
    const Vision_FOLDER = "./AllFile/ImageToAudioFile";
    const files = await fs.readdir(Vision_FOLDER);

    if (!files.length) {
      return res.status(500).json({ success: false, error: "No image found" });
    }

    const imagePath = path.join(Vision_FOLDER, files[0]);
    const data = await fs.readFile(imagePath);
    const base64Image = base64.fromByteArray(data);

    const headers = {
      "Content-Type": "application/json",
      Authorization: `Bearer ${openai.apiKey}`,
    };

    const payload = {
      model: "gpt-4-vision-preview",
      messages: [
        {
          role: "user",
          content: [
            {
              type: "text",
              text: `${descriptionValue} must you will give me this language text ${selectedLanguage}`,
            },
            {
              type: "image_url",
              image_url: { url: `data:image/jpeg;base64,${base64Image}` },
            },
          ],
        },
      ],
      max_tokens: 1200,
    };

    const response = await axios.post("https://api.openai.com/v1/chat/completions", payload, {
      headers: headers,
    });

    const responseData = response.data;
    const finalResult = responseData.choices?.[0]?.message?.content;

    if (!finalResult) {
      return res
        .status(200)
        .json({ success: false, error: "No messages found" });
    }

    const mp3 = await openai.audio.speech.create({
      model: "tts-1",
      voice: "nova",
      input: finalResult,
    });

    const fileName = `${Date.now()}.mp3`;
    const filePath = path.join("./AllFile/ImageToAudofileStore", fileName);
    await fs.writeFile(filePath, Buffer.from(await mp3.arrayBuffer()));
    console.log("File written successfully.");

    const result = await imageToAudioModel.create({
      audio: `/findImageToAudio/${fileName}`,
      createDate: new Date().toISOString(),
      email,
    });

    await userModel.findOneAndUpdate(
      { email },
      {
        $inc: {
          apiUseImageAudioLimit: 1,
        },
      }
    );

    await Promise.all(
      files.map((file) => fs.unlink(path.join(Vision_FOLDER, file)))
    );

    res.status(200).json({ success: true, messages: result });
  } catch (error) {
    console.error(error);
    res
      .status(400)
      .json({ success: false, error: "The text could not be generated" });
  }
};
// const imageToAudioCreate = async (req, res) => {
//   const apiKey = await getOpenAIKey();
//   const openai = new OpenAI({
//     apiKey: apiKey,
//   });

//   try {
//     const { descriptionValue, selectedLanguage } = req.body;
//     const email = req.headers.email;
//     const user = await userModel.findOne({ email });
//     if (!user) {
//       return res.status(400).json({ error: "User not found" });
//     }
//     const userPackage = await packageModel.aggregate([
//       {
//         $match: {
//           packageType: user.plan,
//           packageDuration: user.packageTime,
//         },
//       },
//     ]);
//     if (!userPackage || userPackage.length === 0) {
//       return res
//         .status(500)
//         .json({ error: "User package information not found" });
//     }

//     const selectedPackage = userPackage[0];
//     if (user.apiUseImageAudioLimit >= selectedPackage.imageAudioLimit) {
//       return res
//         .status(402)
//         .json({ error: "Access limit exceeded for image generation" });
//     }
//     const Vision_FOLDER = "./AllFile/ImageToAudioFile";
//     const files = await fs.readdir(Vision_FOLDER);

//     if (!files.length) {
//       return res.status(500).json({ success: false, error: "No image found" });
//     }

//     const imagePath = path.join(Vision_FOLDER, files[0]);
//     const data = await fs.readFile(imagePath);
//     const base64Image = base64.fromByteArray(data);

//     const headers = {
//       "Content-Type": "application/json",
//       Authorization: `Bearer ${openai.apiKey}`,
//     };

//     const payload = {
//       model: "gpt-4-vision-preview",
//       messages: [
//         {
//           role: "user",
//           content: [
//             {
//               type: "text",
//               text: `${descriptionValue} must you will give me this language text ${selectedLanguage}`,
//             },
//             {
//               type: "image_url",
//               image_url: { url: `data:image/jpeg;base64,${base64Image}` },
//             },
//           ],
//         },
//       ],
//       max_tokens: 1200,
//     };

//     const response = await fetch("https://api.openai.com/v1/chat/completions", {
//       method: "POST",
//       headers,
//       body: JSON.stringify(payload),
//     });

//     const responseData = await response.json();
//     const finalResult = responseData.choices?.[0]?.message?.content;

//     if (!finalResult) {
//       return res
//         .status(200)
//         .json({ success: false, error: "No messages found" });
//     }

//     const mp3 = await openai.audio.speech.create({
//       model: "tts-1",
//       voice: "nova",
//       input: finalResult,
//     });

//     const fileName = `${Date.now()}.mp3`;
//     const filePath = path.join("./AllFile/ImageToAudofileStore", fileName);
//     await fs.writeFile(filePath, Buffer.from(await mp3.arrayBuffer()));
//     console.log("File written successfully.");

//     const result = await imageToAudioModel.create({
//       audio: `/findImageToAudio/${fileName}`,
//       createDate: new Date().toISOString(),
//       email,
//     });

//     await userModel.findOneAndUpdate(
//       { email },
//       {
//         $inc: {
//           apiUseImageAudioLimit: 1,
//         },
//       }
//     );

//     await Promise.all(
//       files.map((file) => fs.unlink(path.join(Vision_FOLDER, file)))
//     );

//     res.status(200).json({ success: true, messages: result });
//   } catch (error) {
//     console.error(error);
//     res
//       .status(400)
//       .json({ success: false, error: "The text could not be generated" });
//   }
// };


const imageToAudioTableData = async (req, res) => {
  try {
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const matchStage = {
      $match: {
        $or: [
          { text: searchRgx },
          { audio: searchRgx },
          { email: searchRgx },
          { createDate: searchRgx },
        ],
      },
    };
    const sortStage = {
      $sort: { createDate: -1 },
    };
    const [aggregationResult] = await imageToAudioModel.aggregate([
      matchStage,
      sortStage,
      {
        $match: { email: email },
      },
      {
        $facet: {
          data: [{ $skip: skipRow }, { $limit: perPage }],
          total: [{ $count: "total" }],
        },
      },
    ]);
    const users = aggregationResult.data ? aggregationResult.data : [];
    const totalCount = aggregationResult.total?.[0]?.total || 0;
    res.status(200).json({
      status: "success",
      data: { users, totalCount },
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};
const deleteImageAudioReq = async (req, res) => {
  const id = req.params.id;
  const Query = { _id: id };
  try {
    const data = await imageToAudioModel.deleteOne(Query);
    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Not delete data", error: error });
  }
};

module.exports = {
  imageToAudioCreate,
  imageToAudioTableData,
  deleteImageAudioReq,
};
