const express = require("express");
const openAiModel = require("../DataModel/OpenAISettingModel");
const stripeSettingModel = require("../DataModel/StripeSettingModel");
const paypalSettingModel = require("../DataModel/PayPalSettingModel");
const paypal = require("paypal-rest-sdk");
const stripeModel = require("../DataModel/StripeDataModel");
const stripe = require("stripe");

async function getStripeSecretKey() {
  try {
    const settings = await stripeSettingModel.findOne();
    return settings ? settings.stripeSecretKey : null;
  } catch (error) {
    console.error("Error fetching Stripe secret key:", error);
    return null;
  }
}

async function configurePayPal() {
  try {
    const paypalInfo = await paypalSettingModel.findOne();
    if (!paypalInfo) {
      throw new Error("PayPal settings not found");
    }

    paypal.configure({
      mode: paypalInfo.paypalMode,
      client_id: paypalInfo.paypalClientKey,
      client_secret: paypalInfo.paypalSecretKey,
    });
  } catch (error) {
    console.error("Error configuring PayPal SDK:", error);
    throw error;
  }
}
const payProduct = async (req, res) => {
  configurePayPal();
  const paypalInfo = await paypalSettingModel.findOne();
  const { price, packageDuration, packageType, paymentMethod, stripeIcon } =
    req.body;
  try {
    const create_payment_json = {
      intent: "sale",
      payer: {
        payment_method: "paypal",
      },
      redirect_urls: {
        return_url: paypalInfo.paypalSuccessURL,
        cancel_url: paypalInfo.paypalCancelURL,
      },
      transactions: [
        {
          item_list: {
            items: [
              {
                name: packageType,
                sku: "001",
                price: price,
                currency: "USD",
                quantity: 1,
              },
            ],
          },
          amount: {
            currency: "USD",
            total: price,
          },
          description: "Hat for the best team ever",

          custom: JSON.stringify({
            packageType: packageType,
            packageDuration: packageDuration,
            paymentMethod: paymentMethod,
            stripeIcon: stripeIcon,
          }),
        },
      ],
    };

    const payment = await new Promise((resolve, reject) => {
      paypal.payment.create(create_payment_json, (error, payment) => {
        if (error) {
          reject(error);
        } else {
          resolve(payment);
        }
      });
    });

    for (let i = 0; i < payment.links.length; i++) {
      if (payment.links[i].rel === "approval_url") {
        res.status(200).json({ paymentOfData: payment.links[i].href });
      }
    }
  } catch (error) {
    console.log(error.message);
    res.status(500).send("Internal Server Error");
  }
};

const stripeMethod = async (req, res) => {
  const stripeInfo = await stripeSettingModel.findOne();
  const { price, packageDuration, packageType, paymentMethod, stripeIcon } =
    req.body;
  try {
    const stripeSecretKey = await getStripeSecretKey();
    const stripeClient = stripe(stripeSecretKey);
    const unitAmountCents = price * 100;

    const product = await stripeClient.products.create({
      name: `Selected Package:${packageType}`,
      metadata: {
        packageType: packageType,
        packageDuration: packageDuration,
        paymentMethod: paymentMethod,
        stripeIcon: stripeIcon,
      },
    });

    const priceObject = await stripeClient.prices.create({
      unit_amount: unitAmountCents,
      currency: "usd",
      product: product.id,
      metadata: {
        packageType: packageType,
        packageDuration: packageDuration,
        amountUsd: price,
        paymentMethod: paymentMethod,
        stripeIcon: stripeIcon,
      },
    });

    // Create a checkout session for the payment
    const session = await stripeClient.checkout.sessions.create({
      payment_method_types: ["card"],
      mode: "payment",
      line_items: [
        {
          price: priceObject.id,
          quantity: 1,
        },
      ],
      success_url: stripeInfo.stripeSuccessURL,
      cancel_url: stripeInfo.stripeCancelURL,
      metadata: {
        packageType: packageType,
        packageDuration: packageDuration,
        amountUsd: price,
        paymentMethod: paymentMethod,
        stripeIcon: stripeIcon,
      },
    });

    console.log("Product:", product);
    console.log("Price Object:", priceObject);

    res.json({ id: session.id });
  } catch (error) {
    console.error("Error creating Stripe session:", error);
    res.status(500).json({ error: "Failed to create Stripe session" });
  }
};

const stripeUserData = async (req, res) => {
  try {
    const email = req.headers["email"];
    const data = await stripeModel.find({ email: email });
    if (!data || data.length === 0) {
      return res.status(404).json({ message: "User not found" });
    }

    res.status(200).json(data);
  } catch (error) {
    console.error("Error fetching Stripe user data:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};
const getStripeAdminAllData = async (req, res) => {
  try {
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const searchQuery =
      searchValue !== "0"
        ? {
            $or: [
              { userName: searchRgx },
              { paymentStatus: searchRgx },
              { paymentProcess: searchRgx },
              { packageDuration: searchRgx },
              { packageName: searchRgx },
              { paymentMethod: searchRgx },
              { createDate: searchRgx },
              { email: searchRgx },
            ],
          }
        : {};
    const aggregatePipeline = [
      {
        $match: searchQuery,
      },
      {
        $sort: { createDate: -1 },
      },
      {
        $facet: {
          Total: [{ $count: "count" }],
          Row: [{ $skip: skipRow }, { $limit: perPage }],
        },
      },
    ];
    const [aggResult] = await stripeModel.aggregate(aggregatePipeline);
    const totalCount =
      aggResult.Total.length > 0 ? aggResult.Total[0].count : 0;
    const users = aggResult.Row;
    res.status(200).json({
      status: "success",
      data: { users, totalCount },
    });
  } catch (error) {
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const createOpenAPI = async (req, res) => {
  const { openAiAPI } = req.body;
  try {
    const updateData = {
      openAiAPI: openAiAPI,
    };
    const result = await openAiModel.updateOne({}, updateData, {
      upsert: true,
    });
    res.status(200).json({ message: "Document updated successfully" });
  } catch (error) {
    console.error("Error updating document:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};
const createGoogleClientKey = async (req, res) => {
  const { googleClient } = req.body;
  try {
    const updateData = {
      googleClient: googleClient,
    };

    const result = await googleAuthModel.updateOne({}, updateData, {
      upsert: true,
    });
    res.status(200).json({ message: "Document updated successfully" });
  } catch (error) {
    console.error("Error updating document:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};

//get createOpenAI API for admin
const getOpenAPIAdmin = async (req, res) => {
  try {
    const documents = await openAiModel.find();
    if (documents.length > 0) {
      res
        .status(200)
        .json({ message: "Documents retrieved successfully", data: documents });
    } else {
      res.status(404).json({ message: "No documents found" });
    }
  } catch (error) {
    console.error("Error retrieving documents:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};

//Create paypal setting data API for admin
const createPaypalData = async (req, res) => {
  const {
    paypalClientKey,
    paypalSignature,
    paypalSecretKey,
    paypalSuccessURL,
    paypalCancelURL,
    paypalMode,
  } = req.body;
  try {
    const updateData = {
      paypalClientKey: paypalClientKey,
      paypalSignature: paypalSignature,
      paypalSecretKey: paypalSecretKey,
      paypalSuccessURL: paypalSuccessURL,
      paypalCancelURL: paypalCancelURL,
      paypalMode: paypalMode,
    };
    const result = await paypalSettingModel.updateOne({}, updateData, {
      upsert: true,
    });
    res.status(200).json({ message: "Document updated successfully" });
  } catch (error) {
    console.error("Error updating document:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};

//get paypal data API for admin
const getPaypalDataAdmin = async (req, res) => {
  try {
    const documents = await paypalSettingModel.find();
    if (documents.length > 0) {
      res
        .status(200)
        .json({ message: "Documents retrieved successfully", data: documents });
    } else {
      res.status(404).json({ message: "No documents found" });
    }
  } catch (error) {
    console.error("Error retrieving documents:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};

const createStripeData = async (req, res) => {
  const {
    stripeClientKey,
    stripeSignature,
    stripeSecretKey,
    stripeSuccessURL,
    stripeCancelURL,
  } = req.body;
  try {
    const updateData = {
      stripeClientKey: stripeClientKey,
      stripeSignature: stripeSignature,
      stripeSecretKey: stripeSecretKey,
      stripeSuccessURL: stripeSuccessURL,
      stripeCancelURL: stripeCancelURL,
    };
    const result = await stripeSettingModel.updateOne({}, updateData, {
      upsert: true,
    });
    res.status(200).json({ message: "Document updated successfully" });
  } catch (error) {
    console.error("Error updating document:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};
//get stripe data API for admin
const getStripeDataAdmin = async (req, res) => {
  try {
    const documents = await stripeSettingModel.aggregate([
      {
        $project: {
          stripeClientKey: 1,
          _id: 0,
        },
      },
    ]);

    if (documents.length > 0) {
      res
        .status(200)
        .json({ message: "Documents retrieved successfully", data: documents });
    } else {
      res.status(404).json({ message: "No documents found" });
    }
  } catch (error) {
    console.error("Error retrieving documents:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};

module.exports = {
  payProduct,
  stripeMethod,
  stripeUserData,
  createPaypalData,
  createStripeData,
  createOpenAPI,
  getOpenAPIAdmin,
  getPaypalDataAdmin,
  getStripeDataAdmin,
  getStripeAdminAllData,
  createGoogleClientKey,
};
