const natural = require("natural");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");

const contentDetector = {
  detectContentUniqueness: function (inputText) {
    const tokenizer = new natural.WordTokenizer();
    const words = tokenizer.tokenize(inputText);
    const wordFreqMap = words.reduce((freqMap, word) => {
      freqMap[word] = (freqMap[word] || 0) + 1;
      return freqMap;
    }, {});
    const uniqueWords = new Set(words);
    const uniquenessPercentage = (uniqueWords.size / words.length) * 100;
    let result;
    let matchedContent = [];

    if (uniquenessPercentage >= 70) {
      result = "Unique-Text";
    } else {
      result = "Copy-Text";
      matchedContent = Object.entries(wordFreqMap)
        .filter(([word, freq]) => freq > 1)
        .map(([word, freq]) => ({ word, frequency: freq }));
    }

    return {
      result: result,
      uniquenessPercentage: uniquenessPercentage,
      matchedContent: matchedContent,
    };
  },
};

const PlagiarismChecker = async (req, res) => {
  const { text } = req.body;
  const { email } = req.headers;

  try {
    // Validate input text
    if (!text) {
      return res
        .status(400)
        .json({ error: "Text input is missing in the request body" });
    }

    // Find user
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    // Find user's package
    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);
    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }
    const selectedPackage = userPackage[0];

    // Check plagiarism usage limit
    if (user.apiUsePlagiarismCheck >= selectedPackage.aiPlagiarismCheck) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for plagiarism check" });
    }

    // Detect content uniqueness
    const detectionResult = contentDetector.detectContentUniqueness(text);
    await userModel.findOneAndUpdate(
      { email },
      { $inc: { apiUsePlagiarismCheck: 1 } }
    );

    res.json(detectionResult);
  } catch (error) {
    console.error("Error in plagiarism checker:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};

module.exports = { PlagiarismChecker };
