const fs = require("fs/promises");
const path = require("path");
const OpenAI = require("openai");
const textToAudioModel = require("../DataModel/textToAudio");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");

const openAiApiModel = require("../DataModel/OpenAISettingModel");
async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}

const textToAudio = async (req, res) => {
  const apiKey = await getOpenAIKey();
  const openai = new OpenAI({
    apiKey: apiKey,
  });

  const { inputText, selectedVoice, voiceoverModel } = req.body;
  const email = req.headers.email;
  if (!inputText) {
    return res.status(400).json({ error: "Text is required" });
  }
  try {
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }
    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);
    if (!userPackage || userPackage.length === 0) {
      return res
        .status(500)
        .json({ error: "User package information not found" });
    }
    const selectedPackage = userPackage[0];
    if (user.apiUseAiVoiceoverLimit >= selectedPackage.aiVoiceoverLimit) {
      return res
        .status(402)
        .json({ error: "Access limit exceeded for audio generation" });
    }

    const mp3 = await openai.audio.speech.create({
      model: voiceoverModel,
      voice: selectedVoice,
      input: inputText,
    });
    const fileName = `${Date.now()}.mp3`;
    const filePath = path.join("./AllFile/TextToAudioFile", fileName);
    const buffer = Buffer.from(await mp3.arrayBuffer());
    await fs.writeFile(filePath, buffer);
    await fs.access(filePath);
    const relativePath = `/findAudio/${fileName}`;

    const timestamp = new Date().toISOString();
    await userModel.findOneAndUpdate(
      { email },
      { $inc: { apiUseAiVoiceoverLimit: 1 } }
    );
    const result = await textToAudioModel.create({
      audio: relativePath,
      text: inputText,
      createDate: timestamp,
      email,
      language: selectedVoice,
      model: voiceoverModel,
      voice: selectedVoice,
    });

    res.status(200).json({
      success: true,
      message: "Audio generated and saved to MongoDB.",
      audioId: result._id,
    });
  } catch (err) {
    console.error(err);
    if (err.response && err.response.data && err.response.data.error) {
      res.status(500).json({ error: err.response.data.error });
    } else {
      res.status(500).json({ error: "Error generating and saving audio" });
    }
  }
};

const AudioTableData = async (req, res) => {
  try {
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");

    const matchStage = [
      {
        $match: {
          $and: [
            { email: email },
            {
              $or: [
                { text: searchRgx },
                { audio: searchRgx },
                { model: searchRgx },
                { createDate: searchRgx },
              ],
            },
          ],
        },
      },
    ];

    const sortStage = { $sort: { createDate: -1 } };
    const aggregationResult = await textToAudioModel.aggregate([
      ...matchStage,
      sortStage,
      {
        $facet: {
          data: [{ $skip: skipRow }, { $limit: perPage }],
          total: [{ $count: "total" }],
        },
      },
    ]);

    const AudioData = aggregationResult[0].data || [];
    const totalCount = aggregationResult[0].total?.[0]?.total || 0;
    res.status(200).json({
      status: "success",
      data: { AudioData, totalCount },
    });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};

const textToAudioCountUserDas = async (req, res) => {
  const email = req.headers["email"];
  const currentYear = new Date().getFullYear();

  try {
    const result = await textToAudioModel
      .aggregate([
        {
          $match: {
            email: email,
            createDate: {
              $gte: new Date(`${currentYear}-01-01T00:00:00.000Z`),
              $lt: new Date(`${currentYear + 1}-01-01T00:00:00.000Z`),
            },
          },
        },
        {
          $group: {
            _id: { $month: "$createDate" },
            audio: { $sum: 1 },
          },
        },
      ])
      .exec();
    const monthlyCounts = Array(12).fill(0);
    result.forEach((item) => {
      const monthIndex = item._id - 1;
      monthlyCounts[monthIndex] = item.audio;
    });

    res.status(200).json({
      status: "success",
      monthlyCounts: monthlyCounts,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};
const deleteAudio = async (req, res) => {
  const id = req.params.id;
  const Query = { _id: id };
  try {
    const data = await textToAudioModel.deleteOne(Query);
    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    res
      .status(400)
      .json({ status: "error", message: "Not delete data", error: error });
  }
};

module.exports = {
  textToAudio,
  AudioTableData,
  deleteAudio,
  textToAudioCountUserDas,
};
