// const fs = require("fs");
// const path = require("path");
// const request = require("request");
// const OpenAI = require("openai");
// const { MongoClient } = require("mongodb");
// const userModel = require("../DataModel/AuthDataModel");
// const packageModel = require("../DataModel/packageModel");
// const openAiApiModel = require("../DataModel/OpenAISettingModel");
// const uri = process.env.DATABASE_URL;
// const client = new MongoClient(uri);
// async function getOpenAIKey() {
//   try {
//     const settings = await openAiApiModel.findOne();
//     return settings ? settings.openAiAPI : null;
//   } catch (error) {
//     console.error("Error fetching OpenAI API key:", error);
//     return null;
//   }
// }

// const ImageController = async (req, res) => {
//   const apiKey = await getOpenAIKey();
//   const openai = new OpenAI({
//     apiKey: apiKey,
//   });
//   const {
//     inputValue,
//     selectedResolution,
//     selectedAmount,
//     imageQuality,
//     selectedStyle,
//     selectedLighting,
//     selectedImageMood,
//     selectedImageModel,
//   } = req.body;

//   const email = req.headers.email;
//   const user = await userModel.findOne({ email });
//   if (!user) {
//     return res.status(400).json({ error: "User not found" });
//   }
//   const userPackage = await packageModel.aggregate([
//     {
//       $match: {
//         packageType: user.plan,
//         packageDuration: user.packageTime,
//       },
//     },
//   ]);

//   if (!userPackage || userPackage.length === 0) {
//     return res
//       .status(500)
//       .json({ error: "User package information not found" });
//   }
//   const selectedPackage = userPackage[0];
//   if (user.apiUseTextToImage >= selectedPackage.textToImage) {
//     return res
//       .status(402)
//       .json({ error: "Access limit exceeded for image generation" });
//   }
//   const numberOfImages = parseInt(selectedAmount, 10);
//   const UPLOAD_FOLDER = "./AllFile/Upload-image";
//   try {
//     await client.connect();
//     const collection = client.db("AiGenerated").collection("images");

//     for (let i = 0; i < numberOfImages; i++) {
//       const response = await openai.images.generate({
//         model: selectedImageModel,
//         prompt: `Please give me a picture of ${inputValue} and fallow my instruction ${selectedStyle}, ${selectedLighting}, ${selectedImageMood}`,
//         size: selectedResolution,
//         quality: imageQuality,
//       });
//       const imageUrl = response.data[0].url;
//       const imagePath = `${Date.now()}_${i}.jpg`;
//       const filePath = path.join(UPLOAD_FOLDER, imagePath);
//       const file = fs.createWriteStream(filePath);
//       await new Promise((resolve, reject) => {
//         request
//           .get({ url: imageUrl })
//           .pipe(file)
//           .on("finish", resolve)
//           .on("error", reject);
//       });

//       const imageUrlSaved = `/uploads/${imagePath}`;
//       const timestamp = new Date().toISOString();
//       await collection.insertOne({
//         url: imageUrlSaved,
//         size: selectedResolution,
//         timestamp: timestamp,
//         inputValue: inputValue,
//         email: email,
//       });
//       await userModel.findOneAndUpdate(
//         { email },
//         { $inc: { apiUseTextToImage: 1 } }
//       );
//       console.log(`Inserted image data ${i + 1} of ${numberOfImages}`);
//     }
//     res.status(200).json({
//       success: true,
//       message: `Generated and saved ${numberOfImages} images successfully.`,
//     });
//   } catch (error) {
//     console.error(error);
//     res.status(500).json({
//       success: false,
//       error: "Failed to generate and save images",
//     });
//   }
// };
const fs = require("fs");
const path = require("path");
const axios = require("axios");
const OpenAI = require("openai");
const { MongoClient } = require("mongodb");
const userModel = require("../DataModel/AuthDataModel");
const packageModel = require("../DataModel/packageModel");
const openAiApiModel = require("../DataModel/OpenAISettingModel");

const uri = process.env.DATABASE_URL;
const client = new MongoClient(uri);

async function getOpenAIKey() {
  try {
    const settings = await openAiApiModel.findOne();
    return settings ? settings.openAiAPI : null;
  } catch (error) {
    console.error("Error fetching OpenAI API key:", error);
    return null;
  }
}

const ImageController = async (req, res) => {
  try {
    const apiKey = await getOpenAIKey();
    const openai = new OpenAI({
      apiKey: apiKey,
    });

    const {
      inputValue,
      selectedResolution,
      selectedAmount,
      imageQuality,
      selectedStyle,
      selectedLighting,
      selectedImageMood,
      selectedImageModel,
    } = req.body;

    const email = req.headers.email;
    const user = await userModel.findOne({ email });
    if (!user) {
      return res.status(400).json({ error: "User not found" });
    }

    const userPackage = await packageModel.aggregate([
      {
        $match: {
          packageType: user.plan,
          packageDuration: user.packageTime,
        },
      },
    ]);

    if (!userPackage || userPackage.length === 0) {
      return res.status(500).json({ error: "User package information not found" });
    }

    const selectedPackage = userPackage[0];
    if (user.apiUseTextToImage >= selectedPackage.textToImage) {
      return res.status(402).json({ error: "Access limit exceeded for image generation" });
    }

    const numberOfImages = parseInt(selectedAmount, 10);
    const UPLOAD_FOLDER = "./AllFile/Upload-image";

    await client.connect();
    const collection = client.db("AiGenerated").collection("images");

    for (let i = 0; i < numberOfImages; i++) {
      const response = await openai.images.generate({
        model: selectedImageModel,
        prompt: `Please give me a picture of ${inputValue} and follow my instruction ${selectedStyle}, ${selectedLighting}, ${selectedImageMood}`,
        size: selectedResolution,
        quality: imageQuality,
      });

      const imageUrl = response.data[0].url;
      const imagePath = `${Date.now()}_${i}.jpg`;
      const filePath = path.join(UPLOAD_FOLDER, imagePath);

      const imageResponse = await axios({
        url: imageUrl,
        method: "GET",
        responseType: "stream",
      });

      const writer = fs.createWriteStream(filePath);
      imageResponse.data.pipe(writer);

      await new Promise((resolve, reject) => {
        writer.on("finish", resolve);
        writer.on("error", reject);
      });

      const imageUrlSaved = `/uploads/${imagePath}`;
      const timestamp = new Date().toISOString();

      await collection.insertOne({
        url: imageUrlSaved,
        size: selectedResolution,
        timestamp: timestamp,
        inputValue: inputValue,
        email: email,
      });

      await userModel.findOneAndUpdate(
        { email },
        { $inc: { apiUseTextToImage: 1 } }
      );

      console.log(`Inserted image data ${i + 1} of ${numberOfImages}`);
    }

    res.status(200).json({
      success: true,
      message: `Generated and saved ${numberOfImages} images successfully.`,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      error: "Failed to generate and save images",
    });
  }
};
const imagCountUserDashboard = async (req, res) => {
  const email = req.headers["email"];
  const currentYear = new Date().getFullYear();

  try {
    await client.connect();
    const collection = client.db("AiGenerated").collection("images");
    const result = await collection
      .aggregate([
        {
          $match: {
            email: email,
            timestamp: {
              $regex: `^${currentYear}`,
            },
          },
        },
        {
          $group: {
            _id: { $month: { $toDate: "$timestamp" } },
            Image: { $sum: 1 },
          },
        },
      ])
      .toArray();

    const monthlyCounts = Array(12).fill(0);

    result.forEach((item) => {
      const monthIndex = item._id - 1;
      monthlyCounts[monthIndex] = item.Image;
    });

    res.status(200).json({
      status: "success",
      monthlyCounts: monthlyCounts,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const formatTimestamp = (timestamp) => {
  const date = new Date(timestamp);
  const day = date.getDate().toString().padStart(2, "0");
  const monthNames = [
    "Jan",
    "Feb",
    "Mar",
    "Apr",
    "May",
    "Jun",
    "Jul",
    "Aug",
    "Sep",
    "Oct",
    "Nov",
    "Dec",
  ];
  const month = monthNames[date.getMonth()];
  return `${day} ${month}`;
};
const AiImageList = async (req, res) => {
  const email = req.headers["email"];
  const currentDate = new Date();
  const currentYear = currentDate.getFullYear();
  const currentMonth = currentDate.getMonth() + 1;

  try {
    await client.connect();
    const collection = client.db("AiGenerated").collection("images");
    const result = await collection
      .aggregate([
        {
          $match: {
            email: email,
            timestamp: {
              $regex: `^${currentYear}-${
                currentMonth < 10 ? "0" + currentMonth : currentMonth
              }-\\d{2}T\\d{2}:\\d{2}:\\d{2}.\\d{3}Z$`,
            },
          },
        },
        {
          $addFields: {
            timestamp: {
              $toDate: "$timestamp",
            },
          },
        },
        {
          $sort: { timestamp: -1 },
        },
        {
          $facet: {
            data: [],
            total: [{ $count: "total" }],
            dailyCounts: [
              {
                $group: {
                  _id: {
                    $dateToString: {
                      format: "%m %d",
                      date: { $toDate: "$timestamp" },
                    },
                  },
                  Image: { $sum: 1 },
                },
              },
              { $sort: { _id: -1 } },
            ],
          },
        },
      ])
      .toArray();
    const { data, total, dailyCounts } = result[0];
    const formattedDailyCounts = dailyCounts.map((item) => ({
      ...item,
      _id: formatTimestamp(item._id),
    }));
    res.status(200).json({
      status: "success",
      data: data,
      total: total[0]?.total || 0,
      dailyCounts: formattedDailyCounts || [],
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const AiImageListHomepage = async (req, res) => {
  try {
    await client.connect();
    const collection = client.db("AiGenerated").collection("images");

    const images = await collection
      .aggregate([
        {
          $project: {
            _id: 0,
            url: 1,
            timestamp: {
              $toDate: "$timestamp",
            },
          },
        },
        {
          $sort: { timestamp: -1 },
        },
      ])
      .toArray();

    res.status(200).json({ status: "success", data: images });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: "failed", error: error.message });
  }
};

const AllImageTableData = async (req, res) => {
  try {
    await client.connect();
    const email = req.headers["email"];
    const pageNo = Number(req.params.pageNo);
    const perPage = Number(req.params.perPage);
    const searchValue = req.params.searchKeyword;
    const skipRow = (pageNo - 1) * perPage;
    const searchRgx = new RegExp(searchValue, "i");
    const matchStage = [
      {
        $match: {
          $or: [
            { url: searchRgx },
            { size: searchRgx },
            { email: searchRgx },
            { timestamp: searchRgx },
          ],
        },
      },
      {
        $addFields: {
          timestamp: {
            $toDate: "$timestamp",
          },
        },
      },
    ];

    const sortStage = {
      $sort: { timestamp: -1 },
    };

    const collection = client.db("AiGenerated").collection("images");
    const [aggregationResult] = await collection
      .aggregate([
        ...matchStage,
        sortStage,
        { $match: { email: email } },
        {
          $facet: {
            data: [{ $skip: skipRow }, { $limit: perPage }],
            total: [{ $count: "total" }],
          },
        },
      ])
      .toArray();

    const data = aggregationResult.data || [];
    const totalCount = aggregationResult.total?.[0]?.total || 0;

    res.status(200).json({
      status: "success",
      data: { data, totalCount },
    });
  } catch (error) {
    console.error(error);
    res
      .status(500)
      .json({ status: "failed", error: error.message, data: null });
  }
};

const { ObjectId } = require("mongodb");
const deleteImage = async (req, res) => {
  try {
    const id = req.params.id;
    if (!ObjectId.isValid(id)) {
      return res
        .status(400)
        .json({ status: "error", message: "Invalid ID format" });
    }

    const Query = { _id: new ObjectId(id) };

    await client.connect();
    const collection = client.db("AiGenerated").collection("images");
    const data = await collection.deleteOne(Query);
    await client.close();
    if (data.deletedCount === 0) {
      return res
        .status(404)
        .json({ status: "error", message: "Image not found" });
    }

    res
      .status(200)
      .json({ status: "success", message: "Delete successfully", data: data });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: "error",
      message: "Internal server error",
      error: error.message,
    });
  }
};

module.exports = {
  ImageController,
  AiImageList,
  AiImageListHomepage,
  AllImageTableData,
  deleteImage,
  imagCountUserDashboard,
};
