const multer = require("multer");
const path = require("path");
const ffmpeg = require("fluent-ffmpeg");
const fs = require("fs");
const UPLOAD_FOLDER = "./AllFile/VideoToText";
const convertVideoToMP3 = (videoFilePath, audioFilePath) => {
  return new Promise((resolve, reject) => {
    ffmpeg(videoFilePath)
      .noVideo()
      .audioCodec("libmp3lame")
      .on("end", () => {
        console.log("Video conversion to MP3 complete");
        resolve();
      })
      .on("error", (err) => {
        console.error("Error converting video to MP3:", err);
        reject(err);
      })
      .save(audioFilePath);
  });
};

// Multer storage configuration
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, UPLOAD_FOLDER);
  },
  filename: (req, file, cb) => {
    const audioFileName = `${Date.now()}.mp3`;
    cb(null, audioFileName);
  },
});

// Multer configuration
const videoToText = multer({
  storage: storage,
  limits: {
    fileSize: 50000000, // 5mb
  },
  fileFilter: (req, file, cb) => {
    if (file.mimetype.startsWith("video/")) {
      cb(null, true);
    } else {
      cb(new Error("Only upload supported video files."));
    }
  },
});

videoToText.single("videoFile"),
  async (req, res, next) => {
    try {
      if (!req.file) {
        return res
          .status(400)
          .json({ success: false, error: "No file uploaded." });
      }

      const videoFilePath = req.file.path;
      const audioFileName = `${Date.now()}.mp3`;
      const audioFilePath = path.join(UPLOAD_FOLDER, audioFileName);
      await convertVideoToMP3(videoFilePath, audioFilePath);

      fs.unlinkSync(videoFilePath);

      req.audioFilePath = audioFilePath;
      next();
    } catch (error) {
      console.error("Error processing request:", error);
      return res.status(500).json({
        success: false,
        error: "An error occurred while processing the request",
      });
    }
  };

module.exports = videoToText;
