require("dotenv").config();
const express = require("express");
const Router = require("./Router/app");
const nodemailer = require("nodemailer");
const cors = require("cors");
const helmet = require("helmet");
const hpp = require("hpp");
const rateLimiter = require("express-rate-limit");
const app = express();
const { createOrder, capturePayment } = require("./Utility/Paypal_API.js");
const stripeSettingModel = require("./DataModel/StripeSettingModel");
const stripeModel = require("./DataModel/StripeDataModel");
const stripe = require("stripe");
const SMTPModel = require("./DataModel/SMTPModel");

app.use(helmet());
app.use(cors());
app.use(hpp());
const limiter = rateLimiter({
  windowMs: 60 * 60 * 1000,
  max: 55000,
  message: "Too many requests from this IP, please try again later.",
});
//get stripe SecreteKey
async function getStripeSecretKey() {
  try {
    const settings = await stripeSettingModel.findOne();
    return settings ? settings.stripeSecretKey : null;
  } catch (error) {
    console.error("Error fetching Stripe secret key:", error);
    return null;
  }
}
app.use(limiter);
app.use((req, res, next) => {
  res.setHeader("Access-Control-Allow-Origin", "*");
  res.setHeader("Access-Control-Allow-Methods", "GET, POST, PUT, DELETE");
  res.setHeader("Access-Control-Allow-Headers", "Content-Type");
  res.setHeader("Cross-Origin-Resource-Policy", "cross-origin");
  next();
});

// All backend API routes
app.post("/my-server/create-paypal-order", async (req, res) => {
  try {
    const order = await createOrder(req.body);
    res.json(order);
  } catch (error) {
    console.log(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
});

app.post("/my-server/capture-paypal-order", async (req, res) => {
  const { orderID } = req.body;
  try {
    const captureData = await capturePayment(orderID);
    res.json(captureData);
  } catch (error) {
    console.log(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
});
async function verifyWebhookSignature(headers, webhookEvent) {
  try {
    const verifyEndpoint = "https://api-m.sandbox.paypal.com/v1/notifications/verify-webhook-signature";
    const {
      "paypal-auth-algo": auth_algo,
      "paypal-cert-url": cert_url,
      "paypal-transmission-id": transmission_id,
      "paypal-transmission-sig": transmission_sig,
      "paypal-transmission-time": transmission_time,
    } = headers;
    const webhookId = "4BX97959MR8833324"; // Replace with your PayPal webhook ID

    if (!auth_algo || !cert_url || !transmission_id || !transmission_sig || !transmission_time || !webhookId) {
      console.error("Missing required parameters in headers");
      return false;
    }

    const payload = {
      auth_algo,
      cert_url,
      transmission_id,
      transmission_sig,
      transmission_time,
      webhook_id: webhookId,
      webhook_event: webhookEvent,
    };

    const response = await axios.post(verifyEndpoint, payload, {
      headers: {
        "Content-Type": "application/json",
        Authorization: "Bearer Aa3zhgdKERtB4tcA.kTvAagxGiJIASBadz0w00FJbBSyzhqHgtrgVNAf",
      },
    });

    const verificationResult = response.data;
    console.log("Verification result:", verificationResult);
    return verificationResult.verification_status === "SUCCESS";
  } catch (error) {
    console.error("Error verifying webhook signature:", error.message);
    return false;
  }
}
// Function to verify webhook signature using PayPal's verification endpoint
// async function verifyWebhookSignature(headers, webhookEvent) {
//   try {
//     const verifyEndpoint =
//       "https://api-m.sandbox.paypal.com/v1/notifications/verify-webhook-signature";
//     const {
//       "paypal-auth-algo": auth_algo,
//       "paypal-cert-url": cert_url,
//       "paypal-transmission-id": transmission_id,
//       "paypal-transmission-sig": transmission_sig,
//       "paypal-transmission-time": transmission_time,
//     } = headers;
//     const webhookId = "4BX97959MR8833324"; // Replace with your PayPal webhook ID

//     if (
//       !auth_algo ||
//       !cert_url ||
//       !transmission_id ||
//       !transmission_sig ||
//       !transmission_time ||
//       !webhookId
//     ) {
//       console.error("Missing required parameters in headers");
//       return false;
//     }
//     const payload = {
//       auth_algo,
//       cert_url,
//       transmission_id,
//       transmission_sig,
//       transmission_time,
//       webhook_id: webhookId,
//       webhook_event: webhookEvent,
//     };
//     const response = await fetch(verifyEndpoint, {
//       method: "POST",
//       headers: {
//         "Content-Type": "application/json",
//         Authorization:
//           "Bearer Aa3zhgdKERtB4tcA.kTvAagxGiJIASBadz0w00FJbBSyzhqHgtrgVNAf",
//       },
//       body: JSON.stringify(payload),
//     });

//     if (!response.ok) {
//       throw new Error(
//         `Failed to verify webhook signature: ${response.statusText}`
//       );
//     }

//     const verificationResult = await response.json();
//     console.log("Verification result:", verificationResult);
//     return verificationResult.verification_status === "SUCCESS";
//   } catch (error) {
//     console.error("Error verifying webhook signature:", error.message);
//     return false;
//   }
// }

// Handle incoming PayPal webhook requests
app.post("/paypal/webhook", async (req, res) => {
  const headers = req.headers;
  const webhookEvent = req.body;
  try {
    const isSignatureValid = await verifyWebhookSignature(
      headers,
      webhookEvent
    );
    if (!isSignatureValid) {
      console.error("Invalid PayPal webhook signature");
      return res.status(403).send("Unauthorized");
    }
    console.log("Received valid PayPal webhook:", webhookEvent);
    const { event_type, resource } = webhookEvent;
    const resourceId = resource.id;
    const createTime = resource.create_time;

    switch (event_type) {
      case "PAYMENT.AUTHORIZATION.CREATED":
        console.log(
          "Payment authorization created:",
          resourceId,
          "at",
          createTime
        );
        break;
      default:
        console.log("Unhandled PayPal webhook event type:", event_type);
    }
    res.status(200).send("Webhook Received and Processed");
  } catch (error) {
    console.error("Error processing PayPal webhook:", error.message);
    res.status(500).send("Internal Server Error");
  }
});

// Handle stripe webhook endpoint
app.post(
  "/webhook",
  express.raw({ type: "application/json" }),
  async (req, res) => {
    const smtpInformation = await SMTPModel.findOne();
    const stripeInfo = await stripeSettingModel.findOne();
    const stripeSecretKey = await getStripeSecretKey();
    const stripeClient = stripe(stripeSecretKey);
    const sig = req.headers["stripe-signature"];
    const endpointSecret = stripeInfo.stripeSignature;
    let event;
    try {
      event = stripeClient.webhooks.constructEvent(
        req.body,
        sig,
        endpointSecret
      );
    } catch (err) {
      console.error("⚠️ Webhook signature verification failed.", err.message);
      return res.status(400).send(`Webhook Error: ${err.message}`);
    }
    switch (event.type) {
      case "checkout.session.completed":
        const session = event.data.object;
        console.log("Checkout session completed:", session);
        const {
          customer_details,
          payment_status,
          status,
          currency,
          amount_total,
          metadata,
        } = session;
        const { email, name } = customer_details;
        const { packageType, packageDuration, paymentMethod, stripeIcon } =
          metadata;
        const formattedBasePrice = (amount_total / 100).toFixed(2);
        const newStripeMedia = new stripeModel({
          email: email,
          userName: name,
          paymentStatus: payment_status,
          paymentProcess: status,
          currency: currency,
          paymentPrice: formattedBasePrice,
          packageName: packageType,
          packageDuration: packageDuration,
          paymentMethod: paymentMethod,
          stipeIcon: stripeIcon,
          createDate: new Date(),
        });

        try {
          await newStripeMedia.save();
          console.log("Data saved to database successfully.");
        } catch (error) {
          console.error("Error saving data to database:", error);
        }
        const transporter = nodemailer.createTransport({
          host: smtpInformation.smtpHost,
          port: smtpInformation.smtpPort,
          secure: true,
          auth: {
            user: smtpInformation.smtpSenderEmail,
            pass: smtpInformation.smtpPassword,
          },
        });

        const mailOptions = {
          from: `SmartAI Writing Tools ${smtpInformation.smtpSenderEmail}`,
          to: email,
          subject: "Thanks for your payment!",
          html: `
          <html>
            <head>
              <style>
                /* Define CSS styles for the email */
                body {
                  font-family: Arial, sans-serif;
                  line-height: 1.6;
                  color: #000000;
                }
                .email-container {
                  max-width: 600px;
                  margin: 0 auto;
                  padding: 20px;
                  border: 1px solid #ddd;
                  border-radius: 5px;
                  background-color: #f9f9f9;
                }
                .header {
                  background-color: #007bff;
                  color: #fff;
                  text-align: center;
                  padding: 10px;
                  border-radius: 5px 5px 0 0;
                }
                .content {
                  padding: 20px;
                  background-color: #fff;
                  border-radius: 0 0 5px 5px;
                  color:#000000;
                }
                .thank-you {
                  font-size: 18px;
                  margin-bottom: 20px;
                }
                .details {
                  margin-bottom: 20px;
                  color:#000000;
                }
                .footer {
                  text-align: center;
                  margin-top: 20px;
                  color:#000000;
                }
              </style>
            </head>
            <body>
              <div class="email-container">
                <div class="header">
                  <h2>Thanks for your payment!</h2>
                </div>
                <div class="content">
                  <p class="thank-you">Hello ${name},</p>
                  <p class="thank-you">Thank you for using the SmartAI platform.</p>
                  <div class="details">
                  <p><strong>Payment Method:</strong>${stripeIcon} ${paymentMethod}</p>
                  <p><strong>Package Name:</strong> ${packageType}</p>
                  <p><strong>Package Duration:</strong> ${packageDuration}</p>

                    <p><strong>Payment Status:</strong> ${payment_status}</p>
                    <p><strong>Payment Process Status:</strong> ${status}</p>
                    <p><strong>Total Price:</strong> ${currency} ${formattedBasePrice}</p>
                  </div>
                  <p>We appreciate your business. If you have any questions, please feel free to contact us.</p>
                </div>
                <div class="footer">
                  <p>Thank you,<br>@mernsolution Support Team</p>
                  <p>contact:</strong>hmrubel0143@gmail.com</p>
                </div>
              </div>
            </body>
          </html>
        `,
        };

        try {
          const info = await transporter.sendMail(mailOptions);
          console.log("Email sent:", info.messageId);
        } catch (error) {
          console.error("Error sending email:", error);
        }
        break;
      case "payment_intent.succeeded":
        const paymentIntent = event.data.object;
        console.log("Payment intent succeeded:", paymentIntent);
        break;
      default:
        console.log(`Unhandled event type: ${event.type}`);
    }

    res.status(200).end();
  }
);
app.use(express.json({ limit: "100mb" }));
app.use("/v1", Router);
module.exports = app;
